import pytest
from .. import gpio as g


class Constants:
    INPUT = 0
    OUTPUT = 1
    PUD_OFF = 0
    PUD_DOWN = 1
    PUD_UP = 2

ctes = Constants()


class PiSpy:
    def __init__(self) -> None:
        self.calls = []
        self.modes = {}
        self.read_value = g.HIGH

    def get_mode(self, pin):
        return self.modes[pin]

    def set_mode(self, pin, mode):
        self.calls.append({
            "method": 'set_mode',
            "args": {
                "gpio": pin,
                "mode": mode,
            },
        })
        self.modes[pin] = mode
    
    def set_pull_up_down(self, pin, pullupdown):
         self.calls.append({
            "method": 'set_pull_up_down',
            "args": {
                "gpio": pin,
                "pullupdown": pullupdown,
            },
        })

    
    def write(self, pin, value):
        self.calls.append({
            "method": 'write',
            "args": {
                "gpio": pin,
                "value": value,
            },
        })
    
    def read(self, pin):
        self.calls.append({
            "method": 'read',
            'args': {
                'gpio': pin,
            }
        })
        return self.read_value

def test_set_pin_as_output():
    pi = PiSpy()

    g.set_mode_output(pi, 17, ctes)

    assert len(pi.calls) == 1

    method = pi.calls[0]['method']
    args = pi.calls[0]['args']

    assert method == 'set_mode'
    assert args['gpio'] == 17
    assert args['mode'] == ctes.OUTPUT

testpullupdown = [
    (ctes.PUD_OFF),
    (ctes.PUD_DOWN),
    (ctes.PUD_UP),
]

@pytest.mark.parametrize("pullupdown", testpullupdown)
def test_set_pin_as_input(pullupdown):
    pi = PiSpy()
    pin = 17

    g.set_mode_input(pi, pin, pullupdown, ctes)

    assert len(pi.calls) == 2

    method = pi.calls[0]['method']
    args = pi.calls[0]['args']

    assert method == 'set_mode'
    assert args['gpio'] == pin
    assert args['mode'] == ctes.INPUT

    method = pi.calls[1]['method']
    args = pi.calls[1]['args']

    assert method == 'set_pull_up_down'
    assert args['gpio'] == pin
    assert args['pullupdown'] == pullupdown



def test_cant_write_to_input():
    pi = PiSpy()

    g.set_mode_input(pi, 17, ctes)

    with pytest.raises(g.PinIsNotOutputError):
        g.write_pin(pi, 17, g.HIGH, ctes)


def test_can_write_to_input():
    pi = PiSpy()

    g.set_mode_output(pi, 17, ctes)
    g.write_pin(pi, 17, g.HIGH, ctes)

    assert len(pi.calls) == 2
    
    method = pi.calls[1]['method']
    args = pi.calls[1]['args']

    assert method == 'write'
    assert args['gpio'] == 17
    assert args['value'] == 1


def test_cant_read_from_output():
    pi = PiSpy()

    g.set_mode_output(pi, 17, ctes)

    with pytest.raises(g.PinIsNotInputError):
        g.read_pin(pi, 17, ctes)


def test_can_read_from_input():
    pi = PiSpy()

    g.set_mode_input(pi, 17, ctes)
    read = g.read_pin(pi, 17, ctes)

    assert len(pi.calls) == 3
    
    method = pi.calls[2]['method']
    args = pi.calls[2]['args']

    assert method == 'read'
    assert args['gpio'] == 17
    assert read == pi.read_value